#ifndef __TCollection__
#define __TCollection__

//	===========================================================================

#include "TPointerCollection.hpp"
#include "../Exceptions/CException.hpp"
using Exponent::Exceptions::CException;


namespace Exponent
{
	namespace Collections
	{
		/**
		 * @class TCollection TCollection.hpp
		 * @brief Template'd array of elements
		 *
		 * Stores an intrnal array of objects. For this to work properly, any objects must\n
		 * have overloaded assignment operators
		 *
		 * @date 23/01/2006
		 * @author Paul Chana
		 * @version 1.0.0 Initial version
		 * @version 1.0.1 Added copyBuffer function
		 *
		 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
		 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
		 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
		 * All content is the Intellectual property of Exp Digital Uk.\n
		 * Certain sections of this code may come from other sources. They are credited where applicable.\n
		 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
		 *
		 * $Id: TCollection.hpp,v 1.9 2007/02/08 21:06:44 paul Exp $
		 */
		template<class TypeName> class TCollection : public CCountedObject
		{
			/** @cond */
			EXPONENT_CLASS_DECLARATION;
			/** @endcond */

//	===========================================================================

		public:

//	===========================================================================

			/**
			 * Construction
			 * @param size The size of the array
			 */
			TCollection(const long size);

			/**
			 * Copy construction
			 * @param other The aray to copy
			 */
			TCollection(const TCollection<TypeName> &other);

			/**
			 * Construction
			 */
			TCollection();

			/**
			 * Destruction
			 */
			~TCollection();

//	===========================================================================

			/**
			 * Assignment operator
			 * @param other The vector to copy
			 * @retval TVector& A reference to this
			 */
			TCollection &operator = (const TCollection<TypeName> &other);

			/**
			 * Set all elements to
			 * @param element The element to set all to
			 * @retval TVectory& A reference to this
			 */
			TCollection &operator = (const TypeName &element);

			/**
			 * Get the element at a specific index
			 * @param index The index of the item
			 * @retval TypeName* The object or null on error
			 */
			TypeName *operator [] (const long index);

//	===========================================================================

			/**
			 * Get the element at a specific index
			 * @param index The index of the element
			 * @retval TypeName The value
			 */
			TypeName elementAtIndex(const long index) const;

			/**
			 * Get a pointer to the element at a specific index
			 * @param index The index of the element
			 * @retval TypeName* The object or null on error
			 */
			TypeName *elementPointerAtIndex(const long index);

			/**
			 * Set the value of an element
			 * @param value The value to give the element
			 * @param index The index of the element to set
			 */
			void addElementAtIndex(const TypeName &value, const long index);

//	===========================================================================

			/**
			 * Get the index of an object
			 * @param object The object to find
			 * @retval long The index of the pointer or TPOINTERCOLLECTION_FAILED_TO_FIND_POINTER on error
			 * @see TPointerCollection
			 * @note Does a forward find for the first equal object. Uses overloaded equality operator
			 */
			long getIndexOfObject(const TypeName &object);

			/**
			 * Is pointer in array
			 * @param object The object to look for
			 * @retval bool True if this object is stored in the array, false otherwise
			 * @see TPointerCollection
			 * @note Does a forward find for the first equal object. Uses overloaded equality operator
			 */
			bool isObjectInArray(const TypeName &object);

//	===========================================================================

			/**
			 * Initialise the unit
			 * @param size The size you want the array to be
			 * @note This function deletes all objects in the array
			 */
			void initialise(const long size);

			/**
			 * Get the size of the array
			 * @retval long The size of the array
			 */
			long getArraySize() const;

			/**
			 * Clear the array
			 */
			void clearArray();

//	===========================================================================

			/**
			 * Contact the array (removes the number of entries starrting at the end
			 * @param contractBy The amount ot shrink the array by
			 */
			void contract(const long contractBy);

			/**
			 * Expand the array (adds extra space at the end of the array
			 * @param expandBy The amount of extra space to add
			 */
			void expand(const long expandBy);

//	===========================================================================

			/**
			 * Swap two pointers
			 * @param index1 The index of the first pointer
			 * @param index2 The index of the second pointer
			 */
			void swapIndexes(const long index1, const long index2);

			/**
			 * Set to a specific value
			 * @param value The value to set all the elements to
			 */
			void setAllElementsTo(const TypeName &value);

			/**
			 * Copy a set of elements
			 * @param buffer The buffer to copy the values in to
			 * @param startIndex The start index to copy from
			 * @param size The size of the buffer being passed in
			 */
			void copyElements(TypeName *buffer, const long startIndex, const long size) const;

			/**
			 * Copy another buffer
			 * @param buffer The buffer to copy
			 * @param size The size of the buffer
			 */
			void copyBuffer(const TypeName *buffer, const long size);

//	===========================================================================

			/**
			 * Get internal buffer
			 * @retval const TypeName * The internal buffer
			 */
			const TypeName *getInternalBuffer() const { return m_array; }

			/**
			 * Get the internal buffer
			 * @retval const TypeName * The internal buffer
			 */
			TypeName *getMutableInternalBuffer() { return m_array; }

//	===========================================================================

		protected:

//	===========================================================================

			/**
			 * Free pointers -> deleteds (but not allways) all pointers
			 */
			void freePointers();

//	===========================================================================

			TypeName *m_array;			/**< The internal array */
			long m_arraySize;			/**< The size of the array */
		};

		/** @cond */

		EXPONENT_TEMPLATE_CLASS_IMPLEMENTATION(TCollection<TypeName>, TypeName, CCountedObject);

//	===========================================================================
		template<class TypeName> TCollection<TypeName>::TCollection(const long size) : m_array(NULL), m_arraySize(0)
		{
			// We start with no array
			NULL_POINTER(m_array);
			m_arraySize = 0;

			// Now initialise
			this->initialise(size);
		}

//	===========================================================================
		template<class TypeName> TCollection<TypeName>::TCollection(const TCollection<TypeName> &other) : m_array(NULL), m_arraySize(0)
		{
			// We start with no array
			NULL_POINTER(m_array);
			m_arraySize = 0;

			// Copy it
			*this = other;
		}

//	===========================================================================
		template<class TypeName> TCollection<TypeName>::TCollection() : m_array(NULL), m_arraySize(0)
		{
			NULL_POINTER(m_array);
			m_arraySize = 0;
		}

//	===========================================================================
		template<class TypeName> TCollection<TypeName>::~TCollection()
		{
			this->freePointers();
		}

//	===========================================================================
		template<class TypeName> TCollection<TypeName> &TCollection<TypeName>::operator = (const TCollection<TypeName> &other)
		{
			// If we are not the same
			if (this != &other)
			{
				// Check if the sizes are different
				if (other.m_arraySize != m_arraySize)
				{
					// Delete the other pointers
					this->freePointers();
	
					// Get the size
					this->initialise(other.m_arraySize);
				}

				// Copy the elements
				other.copyElements(m_array, 0, m_arraySize);
			}

			// Return a reference to this
			return *this;
		}

//	===========================================================================
		template<class TypeName> TCollection<TypeName> &TCollection<TypeName>::operator = (const TypeName &element)
		{
			this->setAllElementsTo(element);
			return *this;
		}

//	===========================================================================
		template<class TypeName> TypeName *TCollection<TypeName>::operator [] (const long index)
		{
			return this->elementPointerAtIndex(index);
		}

//	===========================================================================
		template<class TypeName> TypeName TCollection<TypeName>::elementAtIndex(const long index) const
		{
		//	CAssert::assertNotNull(m_array);
		//	assert(index >= 0 && index < m_arraySize);

			//if (m_array && index >= 0 && index < m_arraySize)
			//{
				return m_array[index];
			//}

			// Failed, badly
			//throw CException("No array OR index bounds error", "template<class TypeName> TypeName TCollection<TypeName>::elementAtIndex(const long)");
		}

//	===========================================================================
		template<class TypeName> TypeName *TCollection<TypeName>::elementPointerAtIndex(const long index)
		{
			// If we have a valid array
			if (m_array && index >= 0 && index < m_arraySize)
			{
				return &m_array[index];
			}

			// Failed!
			return NULL;
		}

//	===========================================================================
		template<class TypeName> void TCollection<TypeName>::addElementAtIndex(const TypeName &value, const long index)
		{
			// If we have a valid array
			if (m_array && index >= 0 && index < m_arraySize)
			{
				m_array[index] = value;	
			}
		}

//	===========================================================================
		template<class TypeName> long TCollection<TypeName>::getIndexOfObject(const TypeName &object)
		{
			// If we have an array
			if (m_array)
			{
				// Loop through the array
				for (long i = 0; i < m_arraySize; i++)
				{
					// If the objects are equal
					if (m_array[i] == object)
					{
						return i;
					}
				}
			}

			// Failed!
			return TPointerCollection<TypeName>::TPOINTERCOLLECTION_FAILED_TO_FIND_POINTER;
		}

//	===========================================================================
		template<class TypeName> bool TCollection<TypeName>::isObjectInArray(const TypeName &object)
		{
			return (this->getIndexOfObject(object) != TPointerCollection<TypeName>::TPOINTERCOLLECTION_FAILED_TO_FIND_POINTER);
		}

//	===========================================================================
		template<class TypeName> void TCollection<TypeName>::initialise(const long size)
		{
			// Check that they're not initialising to the same size
			if (size != m_arraySize)
			{
				// Delete any old array that we have
				this->freePointers();

				// Store the size
				m_arraySize = size;

				// Create the array
				m_array = new TypeName[m_arraySize];
			}
		}

//	===========================================================================
		template<class TypeName> long TCollection<TypeName>::getArraySize() const
		{
			return m_arraySize;
		}

//	===========================================================================
		template<class TypeName> void TCollection<TypeName>::clearArray()
		{
			this->freePointers();
		}

//	===========================================================================
		template<class TypeName> void TCollection<TypeName>::contract(const long contractBy)
		{
			// Check its a valid range
			if (m_arraySize - contractBy <= 0)
			{
				throw CException("Conraction causes array to be <= 0 in size", "template<class TypeName> void TCollection<TypeName>::contract(const long)");
			}

			// Store the contraction size
			const long newSize = m_arraySize - contractBy;

			// Make a copy of the array
			TypeName *temp = new TypeName[newSize];

			// Copy the elements
			this->copyElements(temp, 0, newSize);

			// Delete the old array
			this->freePointers();

			// Now store the new pointer
			m_array     = temp;
			m_arraySize = newSize;
		}

//	===========================================================================
		template<class TypeName> void TCollection<TypeName>::expand(const long expandBy)
		{	
			// Store the new size
			const long newSize = m_arraySize + expandBy;

			// Make a copy of the array
			TypeName *temp = new TypeName[newSize];

			// Copy the elements
			this->copyElements(temp, 0, m_arraySize);

			// Delete the old array
			this->freePointers();

			// Now store the new pointer
			m_array     = temp;
			m_arraySize = newSize;
		}

//	===========================================================================
		template<class TypeName> void TCollection<TypeName>::swapIndexes(const long index1, const long index2)
		{
			// If we have validity
			if (m_array && index1 >= 0 && index2 >= 0 && index1 < m_arraySize && index2 < m_arraySize)
			{
				// Copy the pointer
				TypeName object = m_array[index1];

				// Move first pointer
				m_array[index1] = m_array[index2];

				// Store the copy
				m_array[index2] = object;
			}
		}

//	===========================================================================
		template<class TypeName> void TCollection<TypeName>::setAllElementsTo(const TypeName &value)
		{
			// If we have an array
			if (m_array)
			{
				// Loop through
				for (long i = 0; i < m_arraySize; i++)
				{
					// And set the value
					m_array[i] = value;
				}
			}
		}

//	===========================================================================
		template<class TypeName> void TCollection<TypeName>::copyElements(TypeName *buffer, const long startIndex, const long size) const
		{
			// Check we have validity
			if (m_array && buffer && startIndex >= 0 && startIndex < m_arraySize && startIndex + size <= m_arraySize)
			{
				// Loop over range specified
				for (long i = startIndex; i < size; i++)
				{
					buffer[i] = m_array[i];
				}
			}
		}

//	===========================================================================
		template<class TypeName> void TCollection<TypeName>::copyBuffer(const TypeName *buffer, const long size)
		{
			// If we are not the same
			if (buffer)
			{
				// Check if the sizes are different
				if (size != m_arraySize)
				{
					// Delete the other pointers
					this->freePointers();

					// Get the size
					this->initialise(size);
				}

				// We now copy the elements in
				for (long i = 0; i < size; i++)
				{
					m_array[i] = buffer[i];
				}

				for (long i = 0; i < this->getArraySize(); i++)
				{
					if (m_array[i] != buffer[i])
					{
						float a = 1;
						a =2;
					}
				}
			}
		}

//	===========================================================================
		template<class TypeName> void TCollection<TypeName>::freePointers()
		{
			FREE_ARRAY_POINTER(m_array);
			m_arraySize = 0;
		}

		/** @endcond */
	}
}
#endif	// End of TCollection.hpp